// Script to add sample recurring tasks with periods
const { PrismaClient } = require("@prisma/client");
const prisma = new PrismaClient();

async function addSampleTasks() {
  console.log("🌱 Adding sample recurring tasks...");

  try {
    // Get admin user
    const admin = await prisma.user.findFirst({
      where: { role: "ADMIN" },
    });

    if (!admin) {
      console.error("❌ No admin user found. Please run seed first.");
      return;
    }

    // Get technicians
    const technicians = await prisma.user.findMany({
      where: { role: "TECHNICIAN" },
      take: 3,
    });

    if (technicians.length === 0) {
      console.error("❌ No technicians found. Please run seed first.");
      return;
    }

    // Get departments, locations, and elements
    const departments = await prisma.department.findMany({ take: 3 });
    if (departments.length === 0) {
      console.error("❌ No departments found. Please run seed first.");
      return;
    }

    const locations = await prisma.location.findMany({
      where: { departmentId: departments[0].id },
      take: 2,
    });

    const elements = await prisma.element.findMany({
      where: { locationId: locations[0]?.id },
      take: 3,
    });

    // Calculate dates
    const today = new Date();
    today.setHours(0, 0, 0, 0);

    // Start date: 30 days ago for daily tasks, 3 months ago for weekly, 6 months ago for monthly
    // This ensures we have plenty of past periods to show
    const dailyStartDate = new Date(today);
    dailyStartDate.setDate(dailyStartDate.getDate() - 30); // 30 days ago

    const weeklyStartDate = new Date(today);
    weeklyStartDate.setDate(weeklyStartDate.getDate() - 90); // 90 days ago (3 months)

    const monthlyStartDate = new Date(today);
    monthlyStartDate.setMonth(monthlyStartDate.getMonth() - 6); // 6 months ago

    // Sample tasks
    const sampleTasks = [
      {
        title: "Daily Equipment Check - ICU",
        description:
          "Daily inspection of all ICU equipment including ventilators, monitors, and life support systems.",
        isRecurring: true,
        recurrencePattern: "daily",
        startDate: dailyStartDate,
        assignedToId: technicians[0]?.id || null,
        scopeType: "location",
        locationId: locations[0]?.id || null,
        departmentId: departments[0]?.id || null,
      },
      {
        title: "Weekly Maintenance - Radiology Department",
        description:
          "Weekly maintenance check for all radiology equipment including MRI, CT, and X-ray machines.",
        isRecurring: true,
        recurrencePattern: "weekly",
        startDate: weeklyStartDate,
        assignedToId: technicians[1]?.id || null,
        scopeType: "department",
        departmentId: departments[0]?.id || null,
      },
      {
        title: "Monthly Safety Inspection - Emergency Department",
        description:
          "Monthly comprehensive safety inspection of all emergency department equipment.",
        isRecurring: true,
        recurrencePattern: "monthly",
        startDate: monthlyStartDate,
        assignedToId: technicians[0]?.id || null,
        scopeType: "element",
        elementId: elements[0]?.id || null,
        locationId: locations[0]?.id || null,
        departmentId: departments[0]?.id || null,
      },
      {
        title: "Daily Ventilator Check",
        description: "Daily check of ventilator systems in the ICU.",
        isRecurring: true,
        recurrencePattern: "daily",
        startDate: dailyStartDate,
        assignedToId: technicians[0]?.id || null,
        scopeType: "element",
        elementId: elements[0]?.id || null,
        locationId: locations[0]?.id || null,
        departmentId: departments[0]?.id || null,
      },
    ];

    // Create tasks
    const createdTasks = [];
    for (const taskData of sampleTasks) {
      try {
        const task = await prisma.task.create({
          data: {
            ...taskData,
            createdById: admin.id,
            attachmentsJson: "[]",
            selectedLocationsJson: null,
            selectedElementsJson: null,
          },
        });
        createdTasks.push(task);
        console.log(`✅ Created task: ${task.title}`);
      } catch (error) {
        console.error(
          `❌ Failed to create task: ${taskData.title}`,
          error.message
        );
      }
    }

    // Create some period completions for the first daily task
    if (createdTasks.length > 0) {
      const dailyTask = createdTasks.find(
        (t) => t.recurrencePattern === "daily"
      );

      if (dailyTask) {
        console.log("\n📅 Creating period completions...");

        // Create completions for past 20 days (some completed, some not)
        for (let i = 20; i >= 1; i--) {
          const periodDate = new Date(today);
          periodDate.setDate(periodDate.getDate() - i);
          periodDate.setHours(0, 0, 0, 0);

          // Complete about 70% of past periods (skip some randomly)
          if (i % 3 !== 0) {
            try {
              await prisma.taskPeriodCompletion.create({
                data: {
                  taskId: dailyTask.id,
                  periodDate: periodDate,
                  completedAt: new Date(
                    periodDate.getTime() + 10 * 60 * 60 * 1000
                  ), // 10 AM
                  completedById: dailyTask.assignedToId,
                },
              });
              console.log(
                `  ✅ Completed period: ${periodDate.toLocaleDateString()}`
              );
            } catch (error) {
              // Period might already exist, skip
              console.log(
                `  ⚠️  Period already exists: ${periodDate.toLocaleDateString()}`
              );
            }
          } else {
            console.log(
              `  ⏭️  Skipped period (not completed): ${periodDate.toLocaleDateString()}`
            );
          }
        }

        // Create some item completions for today's period
        if (elements.length > 0 && locations.length > 0) {
          const todayPeriod = new Date(today);
          todayPeriod.setHours(0, 0, 0, 0);

          for (let i = 0; i < Math.min(2, elements.length); i++) {
            try {
              await prisma.taskItemCompletion.create({
                data: {
                  taskId: dailyTask.id,
                  elementId: elements[i].id,
                  checkedById: dailyTask.assignedToId || admin.id,
                  status: "OK",
                  checkedAt: new Date(),
                  periodDate: todayPeriod,
                  notes: null,
                  ticketId: null,
                },
              });
              console.log(`  ✅ Completed item: ${elements[i].name}`);
            } catch (error) {
              // Item might already be completed, skip
            }
          }
        }
      }

      // Create completions for the weekly task (past 8 weeks)
      const weeklyTask = createdTasks.find(
        (t) => t.recurrencePattern === "weekly"
      );
      if (weeklyTask) {
        for (let i = 8; i >= 1; i--) {
          const weekDate = new Date(today);
          weekDate.setDate(weekDate.getDate() - i * 7);
          const day = weekDate.getDay();
          const diff = weekDate.getDate() - day + (day === 0 ? -6 : 1);
          const monday = new Date(weekDate.setDate(diff));
          monday.setHours(0, 0, 0, 0);

          // Complete about 75% of past weeks
          if (i % 4 !== 0) {
            try {
              await prisma.taskPeriodCompletion.create({
                data: {
                  taskId: weeklyTask.id,
                  periodDate: monday,
                  completedAt: new Date(
                    monday.getTime() + 2 * 24 * 60 * 60 * 1000
                  ), // 2 days later
                  completedById: weeklyTask.assignedToId,
                },
              });
              console.log(
                `  ✅ Completed weekly period: ${monday.toLocaleDateString()}`
              );
            } catch (error) {
              console.log(
                `  ⚠️  Weekly period already exists: ${monday.toLocaleDateString()}`
              );
            }
          } else {
            console.log(
              `  ⏭️  Skipped weekly period (not completed): ${monday.toLocaleDateString()}`
            );
          }
        }
      }

      // Create completions for the monthly task (past 6 months)
      const monthlyTask = createdTasks.find(
        (t) => t.recurrencePattern === "monthly"
      );
      if (monthlyTask) {
        for (let i = 6; i >= 1; i--) {
          const monthDate = new Date(today);
          monthDate.setMonth(monthDate.getMonth() - i);
          const firstDay = new Date(
            monthDate.getFullYear(),
            monthDate.getMonth(),
            1
          );
          firstDay.setHours(0, 0, 0, 0);

          // Complete about 80% of past months
          if (i % 5 !== 0) {
            try {
              await prisma.taskPeriodCompletion.create({
                data: {
                  taskId: monthlyTask.id,
                  periodDate: firstDay,
                  completedAt: new Date(
                    firstDay.getTime() + 5 * 24 * 60 * 60 * 1000
                  ), // 5 days later
                  completedById: monthlyTask.assignedToId,
                },
              });
              console.log(
                `  ✅ Completed monthly period: ${firstDay.toLocaleDateString()}`
              );
            } catch (error) {
              console.log(
                `  ⚠️  Monthly period already exists: ${firstDay.toLocaleDateString()}`
              );
            }
          } else {
            console.log(
              `  ⏭️  Skipped monthly period (not completed): ${firstDay.toLocaleDateString()}`
            );
          }
        }
      }
    }

    console.log("\n✨ Sample tasks added successfully!");
    console.log(`\n📊 Summary:`);
    console.log(`   - Created ${createdTasks.length} recurring tasks`);
    console.log(
      `   - Daily task: ${
        createdTasks.find((t) => t.recurrencePattern === "daily")?.title
      }`
    );
    console.log(
      `   - Weekly task: ${
        createdTasks.find((t) => t.recurrencePattern === "weekly")?.title
      }`
    );
    console.log(
      `   - Monthly task: ${
        createdTasks.find((t) => t.recurrencePattern === "monthly")?.title
      }`
    );
    console.log(
      `\n💡 You can now view these tasks in the task list and see their period history!`
    );
  } catch (error) {
    console.error("❌ Error adding sample tasks:", error);
    throw error;
  }
}

addSampleTasks()
  .catch((e) => {
    console.error("❌ Failed to add sample tasks:", e);
    process.exit(1);
  })
  .finally(async () => {
    await prisma.$disconnect();
  });
